
# Livre : La régression logistique en épidémiologie
# ----------------------------------------------------------------------------------------------------
# Programme de construction des courbes de modélisation des variables quantitatives 
# (principalement chapitre 4 du livre)
# 
# ----------------------------------------------------------------------------------------------------
#
# Ce programme est appelé par la commande logit_crb() après un modèle de régression logistique avec une seule variable X.
# 
# Les arguments de la fonction sont :
#
# logit_crb(data,mod,reponse,x,x_class=NULL,lab.x,mod,observ=T,courbe=T,logit_min=-6,logit_max=2,title=NULL,c1="black", c2="brown3")
#
# nb : pour les arguments suivis de =, la valeur qui suit est la valeur par défaut
# 
# - data : fichier de données (celui sur lequel le modèle désigné par "mod" a été estimé)
# - mod : modèle logistique pour lequel on veut la courbe de relation entre X et Y
# - reponse : variable Y
# - x : variable X
# - x_class : variable X en classes pour lesquelles on veut marquer les points observés. Il n'est pas nécessaire de spécifier x_class si on ne veut pas représenter les points observés (c'est pourquoi sa valeur par défaut est NULL)
# - lab.x : titre de l'axe des X
# - obs : T ou F selon qu'on veut représenter les points observés ou pas (défaut=T)
# - courbe : T ou F selon qu'on veut représenter la courbe modélisée ou pas (défaut=T)
# - logit_inf et logit_max : ordonnées où doivent être positionnés les points où Logit P n'est pas défini ± infini). Ces valeurs déterminent aussi le cadre de la courbe (si les valeurs prédites par le modèle les dépassent, la courbe est tronquée par le haut ou le bas)
# - title : titre du graphique
# - c1 et c2 : respectivement couleurs de la courbe et des points observés
# ----------------------------------------------------------------------------------------------------
# 
# Exemple
# logit_crb (
#   data = cycles3,
#   reponse = "acc",
#   x = "age",
#   x_class = "agea",
#   mod = fig9,
#   lab.x = "âge en années",
#   obs = T,
#   courbe = F,
#   logit_min = -6,
#   logit_max = +2,
#   title = "Représentation des Logit P observés selon l'âge de la femme en années",
#   c1="red",c2="green"
# )
# ----------------------------------------------------------------------------------------------------
logit_crb <- function(data,reponse,x=NULL,x_class=NULL,mod,lab.x,obs=T,courbe=T,logit_min=-6,logit_max=-2,title=NULL, c1="black", c2="brown3"){

# Vérification de la cohérence de la commande
  if(obs==TRUE & is.null(x_class)){
    stop("Préciser, à l'aide du paramètre x_class, la variable catégorielle selon laquelle vous souhaitez calculer et représenter le logit observé de l'événement Y")
  }
  
  # Valeurs prédites par le modèle
  pred <- predict(mod, se.fit = T)
  val.pred    <- pred$fit
  val.pred.inf <- val.pred  - 1.96*pred$se.fit 
  val.pred.sup <- val.pred  + 1.96*pred$se.fit
  
  data.plot <- data.frame(x = data[!is.na(data[,x]),x], 
                          pred = val.pred, 
                          pred.inf = val.pred.inf, 
                          pred.sup = val.pred.sup)
  
  
  # Valeurs observées par classe de x_class
  classe.x <- data %>%
    group_by(get(x_class)) %>%
    summarize(N=n(), # effectifs par classe
              prop = mean(get(reponse)), # probabilité que Y=1 par classe
              logitP = log(prop/(1-prop))) %>% # logit(P) "moyen" par classe
    ungroup()
  
  names(classe.x) <-  c("x", "N","prop","logitP") 
  
  # Symbole et position des valeurs observées
  classe.x$point <- 16 # forme ronde des points sur le graphique
  

  classe.x$point[classe.x$logitP == -Inf] <- 25 # forme triangle pointe en bas
  classe.x$point[classe.x$logitP == Inf] <- 24 # forme triangle pointe en haut
  
  classe.x$logitP[classe.x$logitP == -Inf] <- logit_min
  classe.x$logitP[classe.x$logitP == Inf] <- logit_max
  
 
  # Valeurs extrèmes de l'axe Y (la courbe peut-être tronquée si elle les dépasse)
  miny<-logit_min-.5
  maxy<-logit_max+.5
  
  # graphique de base
  pbase<-ggplot(data=data) + ylab("Logit P") + xlab(lab.x) +
    theme_classic(base_size = 10) + theme(legend.position="none")+  ggtitle(title) +
    coord_cartesian(ylim = c(miny,maxy))
  
  # Graphique si obs=TRUE
  if (obs==TRUE) {
    p01 <- pbase +
      geom_point (
        data = classe.x,
        aes(x = x, y = logitP, color="Logits observés"),
        pch = classe.x$point,
        size = 2
      ) +
      geom_text (
        data = classe.x,
        aes(
          x = x,
          y = logitP,
          label = ifelse(point %in% c(24, 25), N, '')
        ),
        size = 3,
        hjust = 0,
        vjust = 0,
        nudge_x = -0.30,
        nudge_y = 0.15
      ) +
      scale_y_continuous(breaks = seq(logit_min, logit_max, 2))+
      scale_color_manual(values = c("Logits observés" = c2))   
    
    
    
    # Graphique si obs=TRUE et courbe=FALSE
    if (courbe == FALSE) {
      ptot <- p01
    }
    # Graphique si courbe = TRUE et obs=TRUE
    else if (courbe == TRUE) {
      ptot <- p01 +
        geom_line  (data = data.plot, (aes(x = x, y = val.pred,
                    color="valeurs prédites"))) +
        scale_color_manual(values = c("valeurs prédites" = c1,"Logits observés" = c2))+
        geom_ribbon(data = data.plot,
                    aes(x = x, ymin = pred.inf, ymax = pred.sup),
                    alpha = 0.15)   # alpha = niveau de transparence
    }
  } # fin de la boucle if (obs==TRUE)
    
    # Graphiques si courbe = TRUE et obs=FALSE
    else if (courbe==TRUE & obs==FALSE) {
      ptot<-pbase+
        geom_line  (data = data.plot, aes(x = x, y = val.pred, 
                    color="valeurs prédites")) +
        geom_ribbon(data = data.plot,
                    aes(x = x, ymin = pred.inf, ymax = pred.sup),
                    alpha = 0.15) +  # alpha = niveau de transparence   
        scale_color_manual(values = c("valeurs prédites" = c1))
    } 
    
else {
  ptot<-pbase
  print("pas de courbe")
}
  
  print(ptot)
}
 

